<?php
/**
 * Exchange Resets
 * https://webenginecms.org/
 * 
 * @version 1.2.0
 * @author Lautaro Angelico <http://lautaroangelico.com/>
 * @copyright (c) 2013-2020 Lautaro Angelico, All Rights Reserved
 * @build w3c8c718b75a0f1fa1a557f7f9d70877
 */

namespace Plugin\ExchangeResets;

class ExchangeResets {
	
	private $_exchangeTable = WE_PREFIX . 'WEBENGINE_EXCHANGERESETS';
	
	private $_resets;
	private $_characterData;
	private $_creditConfig = 0;
	private $_exchangeList;
	private $_configFile = 'config.json';
	private $_modulesPath = 'modules';
	private $_sqlPath = 'sql';
	private $_sqlList;
	
	private $_usercpmenu = array(
		array(
			'active' => true,
			'type' => 'internal',
			'phrase' => 'exchangeresets_title',
			'link' => 'exchangeresets/characters',
			'icon' => 'usercp_default.png',
			'visibility' => 'user',
			'newtab' => false,
			'order' => 999,
		),
	);
	
	// CONSTRUCTOR
	
	function __construct() {
		
		// sql tables
		$this->_sqlList = array(
			'WEBENGINE_EXCHANGERESETS' => $this->_exchangeTable
		);
		
		// load configs
		$this->configFilePath = __PATH_EXCHANGERESETS_ROOT__.$this->_configFile;
		if(!file_exists($this->configFilePath)) throw new \Exception(lang('exchangeresets_error_2', true));
		$cfg = file_get_contents($this->configFilePath);
		if(!check_value($cfg)) throw new \Exception(lang('exchangeresets_error_2', true));
		$cfg = json_decode($cfg, true);
		if(!is_array($cfg)) throw new \Exception(lang('exchangeresets_error_2', true));
		
		// check config
		if(!is_array($cfg['exchange_list'])) throw new \Exception(lang('exchangeresets_error_6', true));
		
		// set configs
		$this->_creditConfig = $cfg['credit_config'];
		if(is_array($cfg['exchange_list'])) {
			$this->_exchangeList = $cfg['exchange_list'];
		}
		
		// sql file path
		$this->sqlFilePath = __PATH_EXCHANGERESETS_ROOT__.$this->_sqlPath.'/';
		
		// load database
		$this->db = \Connection::Database('Me_MuOnline');
		
		$this->_checkTables();
	}
	
	// PUBLIC FUNCTIONS
	
	public function loadModule($module) {
		if(!\Validator::Alpha($module)) throw new \Exception(lang('lottery_error_15', true));
		if(!$this->_moduleExists($module)) throw new \Exception(lang('lottery_error_15', true));
		if(!@include_once(__PATH_EXCHANGERESETS_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) throw new \Exception(lang('exchangeresets_error_4', true));
	}
	
	public function setCharacter($character) {
		$Character = new \Character();
		$AccountCharacters = $Character->AccountCharacter($_SESSION['username']);
		if(!is_array($AccountCharacters)) throw new \Exception(lang('error_46',true));
		if(!in_array($character, $AccountCharacters)) throw new \Exception(lang('exchangeresets_error_7',true));
		$characterData = $Character->CharacterData($character);
		
		$this->_characterData = $characterData;
	}
	
	public function setResets($resets) {
		if(!array_key_exists($resets, $this->_exchangeList)) throw new \Exception(lang('exchangeresets_error_8',true));
		$this->_resets = $resets;
	}
	
	public function getExchangeList() {
		return $this->_exchangeList;
	}
	
	public function exchangeResets() {
		if($this->_creditConfig == 0) throw new \Exception(lang('exchangeresets_error_11'));
		
		// load exchanged resets
		$characterExchangedResets = $this->getExchangedResetsFromCharacter($this->_characterData[_CLMN_CHR_NAME_]);
		if(!check_value($characterExchangedResets)) {
			if(!$this->_insertExchangedResetsRecord($this->_characterData[_CLMN_CHR_NAME_])) {
				throw new \Exception(lang('exchangeresets_error_10',true));
			}
			
			$characterExchangedResets = 0;
		}
		
		// available exchanges
		$availableExchanges = $this->_characterData[_CLMN_CHR_RSTS_]-$characterExchangedResets;
		if($availableExchanges < 1) throw new \Exception(lang('exchangeresets_error_9',true));
		if($availableExchanges < $this->_resets) throw new \Exception(lang('exchangeresets_error_9',true));
		
		// credits reward
		$creditsReward = $this->_exchangeList[$this->_resets];
		
		// reward user
		$creditSystem = new \CreditSystem();
		$creditSystem->setConfigId($this->_creditConfig);
		$configSettings = $creditSystem->showConfigs(true);
		switch($configSettings['config_user_col_id']) {
			case 'userid':
				$creditSystem->setIdentifier($_SESSION['userid']);
				break;
			case 'username':
				$creditSystem->setIdentifier($_SESSION['username']);
				break;
			case 'character':
				$creditSystem->setIdentifier($this->_characterData[_CLMN_CHR_NAME_]);
				break;
			default:
				throw new \Exception("Invalid identifier (credit system).");
		}
		$creditSystem->addCredits($creditsReward);
		
		// update exchanged resets
		$this->_updateExchangedResets($this->_characterData[_CLMN_CHR_NAME_], $this->_resets);
		
		// finished
		redirect(1,'exchangeresets/characters/');
	}
	
	public function getExchangedResetsFromCharacter($character) {
		$result = $this->db->query_fetch_single("SELECT * FROM ".$this->_exchangeTable." WHERE character = ?", array($character));
		if(!is_array($result)) return;
		return $result['exchanged_resets'];
	}
	
	public function checkPluginUsercpLinks() {
		if(!is_array($this->_usercpmenu)) return;
		$cfg = loadConfig('usercp');
		if(!is_array($cfg)) return;
		foreach($cfg as $usercpMenu) {
			$usercpLinks[] = $usercpMenu['link'];
		}
		foreach($this->_usercpmenu as $pluginMenuLink) {
			if(in_array($pluginMenuLink['link'],$usercpLinks)) continue;
			$cfg[] = $pluginMenuLink;
		}
		usort($cfg, function($a, $b) {
			return $a['order'] - $b['order'];
		});
		$usercpJson = json_encode($cfg, JSON_PRETTY_PRINT);
		$cfgFile = fopen(__PATH_CONFIGS__.'usercp.json', 'w');
		if(!$cfgFile) throw new \Exception('There was a problem opening the usercp file.');
		fwrite($cfgFile, $usercpJson);
		fclose($cfgFile);
	}
	
	public function getExchangedResetsList() {
		$result = $this->db->query_fetch("SELECT * FROM ".$this->_exchangeTable." ORDER BY exchanged_resets DESC");
		if(!is_array($result)) return;
		return $result;
	}
	
	// PRIVATE FUNCTIONS
	
	private function _insertExchangedResetsRecord($character) {
		$result = $this->db->query("INSERT INTO ".$this->_exchangeTable." (character, exchanged_resets) VALUES (?, ?)", array($character, 0));
		if(!$result) return;
		return true;
	}
	
	private function _updateExchangedResets($character, $exchanged) {
		$result = $this->db->query("UPDATE ".$this->_exchangeTable." SET exchanged_resets = exchanged_resets + ? WHERE character = ?", array($exchanged, $character));
		if(!$result) return;
		return true;
	}
	
	private function _moduleExists($module) {
		if(!check_value($module)) return;
		if(!file_exists(__PATH_EXCHANGERESETS_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) return;
		return true;
	}
	
	private function _checkTables() {
		if(!is_array($this->_sqlList)) return;
		foreach($this->_sqlList as $sqlFile => $tableName) {
			$tableExists = $this->db->query_fetch_single("SELECT * FROM sysobjects WHERE xtype = 'U' AND name = ?", array($tableName));
			if($tableExists) continue;
			if(!$this->_createTable($tableName, $sqlFile)) throw new \Exception(lang('exchangeresets_error_3', true));
		}
	}
	
	private function _createTable($tableName, $sqlFile) {
		if(!array_key_exists($sqlFile, $this->_sqlList)) return;
		if(!file_exists($this->sqlFilePath.$sqlFile.'.txt')) return;
		$query = file_get_contents($this->sqlFilePath.$sqlFile.'.txt');
		if(!check_value($query)) return;
		$queryFinal = str_replace('{TABLE_NAME}', $tableName, $query);
		if(!$queryFinal) return;
		if(!$this->db->query($queryFinal)) return;
		return true;
	}
	
}